import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getInventoryCollection } from "@/lib/database/collections"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const inventoryCollection = await getInventoryCollection()
    const inventory = await inventoryCollection.findOne({ _id: new ObjectId(params.id) })

    if (!inventory) {
      return NextResponse.json({ success: false, error: "Inventory record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: inventory })
  } catch (error) {
    console.error("Error fetching inventory:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch inventory record" }, { status: 500 })
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body = await request.json()

    const updateData = {
      date: new Date(body.date),
      branchId: new ObjectId(body.branchId),
      type: body.type, // 'consumption' or 'inventory'
      kurczak: body.kurczak,
      wolowina: body.wolowina,
      kraftowe: body.kraftowe,
      majonez: body.majonez,
      ketchup: body.ketchup,
      gas: body.gas,
      notes: body.notes,
      updatedAt: new Date(),
    }

    const inventoryCollection = await getInventoryCollection()
    const result = await inventoryCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "Inventory record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("Error updating inventory:", error)
    return NextResponse.json({ success: false, error: "Failed to update inventory record" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const inventoryCollection = await getInventoryCollection()
    const result = await inventoryCollection.deleteOne({ _id: new ObjectId(params.id) })

    if (result.deletedCount === 0) {
      return NextResponse.json({ success: false, error: "Inventory record not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, message: "Inventory record deleted successfully" })
  } catch (error) {
    console.error("Error deleting inventory:", error)
    return NextResponse.json({ success: false, error: "Failed to delete inventory record" }, { status: 500 })
  }
}
